package maple;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.TreeMap;

import com.maplesoft.externalcall.MapleException;
import com.maplesoft.openmaple.Algebraic;
import com.maplesoft.openmaple.AlgebraicRTable;

import edu.buffalo.fs7.mathlib.Complex;
import edu.buffalo.fs7.mathlib.Gate;
import edu.buffalo.fs7.mathlib.Mathlib;
import edu.buffalo.fs7.mathlib.Matrix;

public class KetUtilities extends LinkedHashSet{

	private static final long serialVersionUID = -413199198672591797L;

	/**
	 * Applys a controlloed Matrix Operation on a state
	 * @param controllQubits The Qubits that work as a switch
	 * @param effectedQubits The Qubits that will be manipulated by the matrix
	 * @param kets The state to manipulate
	 * @param matrix The transformation matrix
	 * @return The result of matrix application
	 */
	
	public static LinkedHashSet<Ket> applyMatrix(LinkedList<Integer> controllQubits, LinkedList<Integer> effectedQubits,LinkedHashSet<Ket> kets, String[][] matrix) {
		LinkedHashSet<Ket> result = new LinkedHashSet<Ket>();
		TreeMap<String, Ket> merger = new TreeMap<String, Ket>();
		if (effectedQubits.size() != 0) {
			Iterator it = kets.iterator();
			while (it.hasNext()) {
				Ket workingKet = (Ket)it.next();
				if (checkBitsSet(controllQubits,workingKet )) {	
					Algebraic temp = ketToStringArray(effectedQubits, workingKet);
					Algebraic temp2 = applyMatrix(temp, matrix);
				
					LinkedHashSet temp3 = algebraicToKet(temp2);
					LinkedHashSet temp4 = upSizeKet(effectedQubits, workingKet, temp3);
					Iterator iter = temp4.iterator();
					while (iter.hasNext()) {
						Ket ket = (Ket)iter.next();
						if (merger.containsKey(makeComparable(ket.getQubitSequenz()))) {
							Ket tempKet = mergeKets(merger.get(makeComparable(ket.getQubitSequenz())), ket);
							if (tempKet.getFrontFactor().equals("0")) {
								merger.remove(makeComparable(ket.getQubitSequenz()));
							} else {
								merger.put(makeComparable(ket.getQubitSequenz()),tempKet);
							}
						
						} else {
							// Remove Kets with 0 as frontFactor
							if (!ket.getFrontFactor().equals("0")) {
								merger.put(makeComparable(ket.getQubitSequenz()),ket);
							}
						}
					}
				} else {
					// We have to insert the unmodified Ket
					result.add(workingKet.copy());
				}
			}
		}
		result.addAll(merger.values());
		return result;		
	}
	
	/**
	 * Merges two Kets. I.e. only the frontFactors are added
	 * Do not use if the QubitSequence is different 
	 * @param a Ket a
	 * @param b Ket b
	 * @return The resulting Ket
	 */
	
	public static Ket mergeKets(Ket a, Ket b) {
		Ket result = new Ket();
		result.setQubitSequenz(a.getQubitSequenz());
		String calc;
		try {
			calc = MapleConnector.mapleEngine.evaluate("(" + a.getFrontFactor() + ") + ("+ b.getFrontFactor() + "):").toString();
			result.setFrontFactor(calc);
		} catch (MapleException e) {
			System.out.println(e.getMessage());
		}
		return result;
	}
	
	/**
	 * Transforms an int[] key for a Ket in a String in order to make
	 * Kets comparable 
	 * @param key The int[] key
	 * @return The String representation of the key
	 */
	private static String makeComparable(int[] key) {
		String result = "";
		for (int i = 0; i< key.length; i++) {
			result += key[i];
		}
		return result;
	}
	
	/**
	 * 
	 * @param effectedQubits
	 * @param originalKet
	 * @param ketsToUpsize
	 * @return
	 */
	
	public static LinkedHashSet<Ket> upSizeKet(LinkedList effectedQubits, Ket originalKet, LinkedHashSet ketsToUpsize){
		LinkedHashSet<Ket> result = new LinkedHashSet<Ket>();
		Iterator it = ketsToUpsize.iterator();
		while (it.hasNext()) {
			Ket workingKet = (Ket)it.next();
			Ket temp = new Ket();
			int[] modifier = new int[originalKet.getQubitSequenz().length];
			int pos = 0;
			temp.setFrontFactor(workingKet.getFrontFactor());
			for (int i=0;i< originalKet.getQubitSequenz().length;i++) {
				if (effectedQubits.contains(new Integer(i))) {
					modifier[i] = workingKet.getQubitSequenz()[pos];
					pos++;
				} else {
					modifier[i] = originalKet.getQubitSequenz()[i];
				}
			}
			temp.setQubitSequenz(modifier);
			result.add(temp);
			
		}
		return result;
	}
	
	/**
	 * This method is used to transform a Maple Alebraic vector object into a LinkedHastSet. 
	 * @param vector
	 * @return
	 */
	
	public static LinkedHashSet algebraicToKet (Algebraic vector) {
		LinkedHashSet<Ket> lHS = new  LinkedHashSet<Ket>();
		int index[] = new int[1];
		
		
		AlgebraicRTable rTable = (AlgebraicRTable) vector;
		
		try {
			int temp =  rTable.upperBound(1) - rTable.lowerBound(1) + 1;
			int qubits = Integer.parseInt(MapleConnector.mapleEngine.evaluate("log[2](" + temp + "):").toString());
			for (int i=0; i < temp; i++ ) {
				index[0] = i+1;
				if (rTable.select(index).toString() != "0") {
					String leadingZeros = "";
					String binaryRepresentation = Integer.toBinaryString(i);
					for (int j = 0; j < qubits - binaryRepresentation.length(); j++) {
						leadingZeros+="0";
					}
					binaryRepresentation = leadingZeros + binaryRepresentation;
					int[] modifier = new int[qubits];
					for (int j = 0; j< qubits; j++) {
						if (binaryRepresentation.charAt(j) == '0') {
							modifier[j] = 0;
						} else {
							modifier[j] = 1;
						}
					}
					//We only add Kets that dont have 0 as their frontFactor
					if (!rTable.select(index).toString().equals("0")) {
						Ket ket = new Ket();
						ket.setFrontFactor(rTable.select(index).toString());
						ket.setQubitSequenz(modifier);
						lHS.add(ket);
					}
				}
			}

		} catch (NumberFormatException e) {
			System.out.println(e.getMessage());
		} catch (MapleException e) {
			System.out.println(e.getMessage());
		}
		return lHS;
		
	}
	
	/**
	 * Applies a matrix on a Maple vector
	 * @param alg The Maple vector
	 * @param matrix The matrix to apply
	 * @return
	 */
	public static Algebraic applyMatrix(Algebraic alg, String[][] matrix) {
		try {
			//System.out.println("evalm((" + alg.toString() + ").(" + getMatrix(matrix,"1") + ")):");
			Algebraic result = MapleConnector.mapleEngine.evaluate("convert(evalm((" + alg.toString() + ").(" + getMatrix(matrix,"1")+")), Vector):");
			return result;
		} catch (MapleException e) {
			System.out.println(e.getMessage());
			return null;
		}
	}
	
	/**
	 * Constructs a Maple matrix to a given String[][] Matrix
	 * 
	 * @param matrix
	 * @param frontFactor
	 * @return
	 */
	public static String getMatrix(String[][] matrix, String frontFactor) {
		String result = frontFactor + "*<";
		for (int i = 0; i < matrix.length; i++) {
			for (int j = 0; j < matrix[i].length; j++) {
				if (matrix[i][j] == null)
					matrix[i][j] = "0";
				if (j == 0) {
					result += "<" + matrix[i][j] + " | ";
				} else if (j == matrix[i].length - 1) {
					result += matrix[i][j] + " > ";
				} else {
					result += matrix[i][j] + " | ";
				}
			}
			if (i < matrix.length - 1) {
				result += " , ";
			}
		}
		result += ">";
		return result;
	}
	
	/**
	 * Transforms a Ket to A String Array
	 * @param effectedQubits
	 * @param ket
	 * @return
	 */
	
	public static Algebraic ketToStringArray(LinkedList effectedQubits, Ket ket) {
		String[][] result = null;
		Iterator it = effectedQubits.iterator();
		
		while (it.hasNext()) {
			int position = (Integer)it.next();
			if (result == null) {
				result = new String[1][2];
				if (ket.getQubitSequenz()[position] == 0) {
					result[0][0] = "1";
					result[0][1] = "0";
				} else {
					result[0][0] = "0";
					result[0][1] = "1";					
				}
			} else {
				String[][] temp = new String[1][2];
				if (ket.getQubitSequenz()[position] == 0) {
					temp[0][0] = "1";
					temp[0][1] = "0";
				} else {
					temp[0][0] = "0";
					temp[0][1] = "1";					
				}
				result = kronProd(result,temp);
			}
			
		}
		Algebraic alg;
		try {
			alg = MapleConnector.mapleEngine.evaluate("convert(evalm(("+ ket.getFrontFactor() + ")*(" + toMapleString(result[0])+")),Vector):");
			
			return alg;
		} catch (MapleException e) {
			System.out.println(e.getMessage());
			return null;
		}
		
	}
	
	/**
	 * Converts a String array in a Maple vector
	 * @param toConv
	 * @return
	 */
	
	public static String toMapleString(String[] toConv) {
		String result = "[";
		for (int i = 0; i < toConv.length - 1 ; i++) {
			result += toConv[i] +  ", ";
		}
		result += toConv[toConv.length - 1] + "]";
		return result;
	}
	
	/**
	 * Calculates the Kronecker product of two matrices
	 * @param m1 matrix1
	 * @param m2 matrix2
	 * @return The resulting matrix
	 */
	
    public static String[][] kronProd(String[][] m1 , String[][] m2) {
    	String[][] result = new String[m1.length*m2.length][m1[0].length*m2[0].length ];
    	for (int i=0; i< result.length; i++) {
    		for (int j=0; j< result[0].length; j++) {
    			result [i][j] = "(" + m1[(int)(i/m2.length)][(int)j/m2[0].length] + ")*("+ m2[(int)(i%m2.length)][(int)(j%m2[0].length)]+")";
    		}
    	}
    	
    	return result;
    	                                                  
    }
    
    /**
     * Prints a state in the form 1*|000> + ... 
     * @param lhs the state to preint
     * @return the string representation
     */
    
    public static String printState(LinkedHashSet<Ket> lhs) {
    	String result = "";
    	Iterator it = lhs.iterator();
    	// For optical reasons we dont want a '+' before the first
    	// Ket
    	if (it.hasNext()) {
    		result += ((Ket)it.next()).toString();
    	}
    	while (it.hasNext()) {
    		result += " + " + ((Ket)it.next()).toString();
    	}
    	return result;
    }
    
    public static void getOperation(Gate container) {
    	System.out.println(container.gate_descr);
    	System.out.println(container.toString());
    	System.out.println(container.getMatrix());
    	System.out.println(container.matrixName);
    	System.out.println(container.toParseableString());

    }
    
    /**
     * The default matrices that are used very often are built
     * into the programm
     * @param indetifier unique String for the matrix
     * @return matrix
     */
    
    public static String[][] getBuiltInMatrix(String identifier) {
    	String[][] matrix = null;
    	if (identifier.equals("H")) {
    		matrix = new String[2][2];
    		matrix[0][0] = "1/sqrt(2)";
    		matrix[0][1] = "1/sqrt(2)";
    		matrix[1][0] = "1/sqrt(2)";
    		matrix[1][1] = "-1/sqrt(2)";
    	} else if (identifier.equals("sigma_x")) {
    		matrix = new String[2][2];
    		matrix[0][0] = "0";
    		matrix[0][1] = "1";
    		matrix[1][0] = "1";
    		matrix[1][1] = "0";
    	} else if (identifier.equals("sigma_y")) {
    		matrix = new String[2][2];
    		matrix[0][0] = "0";
    		matrix[0][1] = "i";
    		matrix[1][0] = "-i";
    		matrix[1][1] = "0";
    	} else if (identifier.equals("sigma_z")) {
    		matrix = new String[2][2];
    		matrix[0][0] = "1";
    		matrix[0][1] = "0";
    		matrix[1][0] = "0";
    		matrix[1][1] = "-1";
    	} else if (identifier.startsWith("R_")) {
    		matrix = new String[2][2];
    		matrix[0][0] = "1";
    		matrix[0][1] = "0";
    		matrix[1][0] = "0";
    		matrix[1][1] = "exp(I*Pi/(2^"+ (Integer.valueOf(identifier.substring(2,identifier.length()))-1)+"))";
    	} else if (identifier.startsWith("S_")) {
    		matrix = new String[2][2];
    		matrix[0][0] = "1";
    		matrix[0][1] = "0";
    		matrix[1][0] = "0";
    		matrix[1][1] = "exp(-I*Pi/(2^"+ (Integer.valueOf(identifier.substring(2,identifier.length()))-1)+"))";
    	} else if (identifier.startsWith("Rz(")) {
    		matrix = new String[2][2];
    		String param = identifier.substring(3,identifier.length()-1);
    		param = param.replaceAll("pi","Pi");
    		matrix[0][0] = "exp(-I*" + param +"/2)";
    		matrix[0][1] = "0";
    		matrix[1][0] = "0";
    		matrix[1][1] = "exp(I*" + param +"/2)";
    	} else if (identifier.startsWith("Rx(")) {
    		matrix = new String[2][2];
    		String param = identifier.substring(3,identifier.length()-1);
    		param = param.replaceAll("pi","Pi");
    		matrix[0][0] = "cos(" + param +  "/2)";
    		matrix[0][1] = "I*sin(" + param + "/2)";
    		matrix[1][0] = "I*sin(" + param + "/2)";
    		matrix[1][1] = "cos(" + param +  "/2)";
    	} else if (identifier.startsWith("Ry(")) {
    		matrix = new String[2][2];
    		String param = identifier.substring(3,identifier.length()-1);
    		param = param.replaceAll("pi","Pi");
    		matrix[0][0] = "cos(" + param +  "/2)";
    		matrix[0][1] = "sin(" + param + "/2)";
    		matrix[1][0] = "-sin(" + param + "/2)";
    		matrix[1][1] = "cos(" + param +  "/2)";
    	}
    	return matrix;
    }
    
    /**
     * Exchanges Qubits a and b
     * @param a Qubit a
     * @param b Qubit b
     * @param state The state to modifiy
     * @return The resulting state
     */
    
    public static LinkedHashSet<Ket> exchangeQubits(int a, int b, LinkedHashSet<Ket> state ) {
    	LinkedHashSet<Ket> result = new LinkedHashSet<Ket>();
    	Iterator<Ket> it = state.iterator();
    	while (it.hasNext()) {
    		Ket workingKet = it.next();
    		Ket toAdd = new Ket();
    		toAdd.setFrontFactor(workingKet.getFrontFactor());
    		int[] modifier = new int[workingKet.getQubitSequenz().length];
    		for (int i=0;i<workingKet.getQubitSequenz().length;i++) {
    			modifier[i] =  workingKet.getQubitSequenz()[i];
    		}
    		// swap
    		modifier[a] = workingKet.getQubitSequenz()[b];
    		modifier[b] = workingKet.getQubitSequenz()[a];
    		toAdd.setQubitSequenz(modifier);
    		result.add(toAdd);
    		
    	}
    	return result;
    }
    
    /**
     * Checks whether the controll bits are set
     * @param controllQubits The Qubits that shall be controlled 
     * @param ket The Ket to check
     * @return false/true
     */
    public static boolean checkBitsSet(LinkedList<Integer> controllQubits, Ket ket) {
    	boolean result = true;
    	Iterator<Integer> it = controllQubits.iterator();
    	while (it.hasNext()) {
    		int pos = it.next().intValue();
    		boolean temp = ket.getQubitSequenz()[pos] != 0;
    		if (temp) {
    			result &= temp;
    		} else {
    			return false;
    		}
    	}
    	return result;
    }
    
    /**
     * 
     * @param controllQubits
     * @param effectedQubit
     * @return
     */
    public static LinkedHashSet<Ket> cNot(LinkedList<Integer> controllQubits, int effectedQubit, LinkedHashSet<Ket> state) {
    	LinkedHashSet<Ket> result = new LinkedHashSet<Ket>();
    	Iterator<Ket> it = state.iterator();
    	while (it.hasNext()) {
    		Ket workingKet = it.next();
    		Ket temp = workingKet.copy();
    		if (checkBitsSet(controllQubits,workingKet )) {	
    			if (temp.getQubitSequenz()[effectedQubit] == 0) {
    				temp.getQubitSequenz()[effectedQubit] = 1;
    			} else {
    				temp.getQubitSequenz()[effectedQubit] = 0;
    			}
    		}
    		result.add(temp);
    		
    	}
    	return result;
    }
    
    /**
     * Perfoms a controlled Phase shift
     * @param controllQubits
     * @param effectedQubit
     * @param state
     * @param phase
     * @return
     */
    public static LinkedHashSet<Ket> cPhase(LinkedList<Integer> controllQubits, int effectedQubit, LinkedHashSet<Ket> state, String phase) {
    	LinkedHashSet<Ket> result = new LinkedHashSet<Ket>();
    	Iterator<Ket> it = state.iterator();
    	while (it.hasNext()) {
    		Ket workingKet = it.next();
    		Ket temp = workingKet.copy();
    		if (checkBitsSet(controllQubits,workingKet )) {
    			try {
    				Algebraic newFrontFactor = MapleConnector.mapleEngine.evaluate(workingKet.getFrontFactor() + " * (exp(Pi*I*" + phase + ")):");
    				temp.setFrontFactor(newFrontFactor.toString());
    			} catch (MapleException e) {
    				System.out.println(e.getMessage());
    			}
    		}
    		
    		result.add(temp);
    	}
    	return result;
    }
    
    /**
     * This function parses a gate and returns an OperationContainer that contains all required information
     * @param gate The gate to analyse
     * @return OperationContainer
     */
    
    public static OperationContainer parseOperation(Gate gate, Matrix errorMatrix) {
    	if ((int)((Complex)Mathlib.getVar("MAPLEDEBUGLEVEL")).re() > 0) {
    		// If the debuglevel is > 0 we print some debug information
        	getOperation(gate);
    	}
    	
    	OperationContainer result = new OperationContainer();
    	LinkedList<Integer> controllQubits = new LinkedList<Integer>();
    	LinkedList<Integer> affectedQubits = new LinkedList<Integer>();
    	String[][] matrix = null;
    	String operation = "";
    	
    	String input = gate.toString();
    	// remove "{" and "}"
    	input = input.substring(1,input.length()-1);
    	// split along ":"
    	String[] workingStrings = input.split(":");
    	
    	
    	// Determine the controll Qubits
    	for (int i = 0; i< workingStrings.length; i++) {
    		if (workingStrings[i].equals("1")){
    			controllQubits.add(new Integer(i));
    		} else if (workingStrings[i].equals("NOT")) {
    			affectedQubits.add(new Integer(i));
    			operation = "NOT";
    		} else if (workingStrings[i].startsWith("Ph")) {
    			operation = "Ph";
    			//  no real matrix 
    			matrix = new String[1][1];
    			String temp = workingStrings[i].split("\\*")[1];
    			temp = temp.replaceAll("\\(","");
    			matrix[0][0] = temp.replaceAll("\\)","");			
    			
    		} else if (workingStrings[i].equals("H")) {
    			operation = "H";
    			matrix = getBuiltInMatrix("H");
    			affectedQubits.add(new Integer(i));
    		} else if (workingStrings[i].equals("sigma_x")) {
    			affectedQubits.add(new Integer(i));
    			operation = "sigma_x";
    			matrix = getBuiltInMatrix("sigma_x");
    		} else if (workingStrings[i].equals("sigma_y")) {
    			affectedQubits.add(new Integer(i));
    			operation = "sigma_y";
    			matrix = getBuiltInMatrix("sigma_y");
    		} else if (workingStrings[i].equals("sigma_z")) {
    			affectedQubits.add(new Integer(i));
    			operation = "sigma_z";
    			matrix = getBuiltInMatrix("sigma_z");
    		} else if (workingStrings[i].equals("xChange")){
    			// Exchanges two Qubits
    			affectedQubits.add(new Integer(i));
    			operation = "xChange";
    		} else if (workingStrings[i].startsWith("R_") || workingStrings[i].startsWith("S_") ||
    				 workingStrings[i].startsWith("Rx(") ||  workingStrings[i].startsWith("Ry(")  
    				 || workingStrings[i].startsWith("Rz(")) {
    			affectedQubits.add(new Integer(i));
    			operation = workingStrings[i];
    			matrix = getBuiltInMatrix(operation);
    		} else if (workingStrings[i].equals("!")) {
    			// partial measurement
    			affectedQubits.add(new Integer(i));
    			operation = "!";
    		} else if (workingStrings[i].equals("-")) {
    			// nothing to do here
    		} else {
    			// We are now in the case of a custom Matrix
    			//FIXME: We need to assure that exact values are saved to the file!
    			operation="customMatrix";
    			matrix = ((Matrix)Mathlib.getVar(gate.matrixName)).getExactContent();
    			affectedQubits.add(new Integer(i));
    		}
    	} 
    	
    	result.setControllQubits(controllQubits);
    	result.setAffectedQubits(affectedQubits);
    	result.setMatrix(applyErrorMatrix(matrix, errorMatrix));
    	result.setOperation(operation);
    	return result;
    }
    
    /**
     * Applys an errorMatrix to a 2x2 dimensonal Matrix
     * @param matrix
     * @param errorMatrix
     * @return
     */
    
    public static String[][] applyErrorMatrix(String[][] matrix, Matrix errorMatrix) {
    	//FIXME: We only apply noise to two dimensional matrices 
    	if (errorMatrix != null && matrix != null && matrix.length == 2 && matrix[0].length == 2) {
    		String[][] result = new String[2][2];
    		result[0][0] = "(" + matrix[0][0] +  ")*((" + errorMatrix.getElement(0,0).re()+ ")+(" + errorMatrix.getElement(0,0).im()+ ")*I" + ")";
    		result[0][1] = "(" + matrix[0][1]+  ")*((" + errorMatrix.getElement(0,1).re()+ ")+(" + errorMatrix.getElement(0,1).im()+ ")*I"+ ")";
    		result[1][0] = "(" + matrix[1][0]+  ")*((" + errorMatrix.getElement(1,0).re()+ ")+(" + errorMatrix.getElement(1,0).im()+ ")*I"+ ")";
    		result[1][1] = "(" +  matrix[1][1]+  ")*((" + errorMatrix.getElement(1,1).re()+ ")+(" + errorMatrix.getElement(1,1).im()+ ")*I"+ ")";
    		return result;
    	} else {
    		return matrix;
    	}
    }
    
    /**
     * This function invokes the needed Operations for one Gate
     * @param con The OperationContainer
     * @return the result of the Operation
     */
    public static LinkedHashSet<Ket> invokeOperation(OperationContainer con, LinkedHashSet<Ket> state) {
    	LinkedHashSet<Ket> result = null;
    	String operation = con.getOperation();
    	if (operation.equals("NOT")) {
    		result = cNot(con.getControllQubits(),con.getAffectedQubits().get(0),state);
    	} else if (operation.equals("H")) {
    		result = applyMatrix(con.getControllQubits(), con.getAffectedQubits(),state, con.getMatrix());
    	} else if (operation.equals("Ph")) {
    		result = cPhase(con.getControllQubits(),1,state,con.getMatrix()[0][0]);
    	} else if (operation.equals("")) {
    		//identity
    		result = state;		
    	} else if (operation.startsWith("sigma")) {
    		// we use the same handler for every sigma_...
    		result = applyMatrix(con.getControllQubits(), con.getAffectedQubits(),state, con.getMatrix());
    	} else if (operation.equals("customMatrix")) {
    		result = applyMatrix(con.getControllQubits(), con.getAffectedQubits(),state, con.getMatrix());
    	}  else if (operation.equals("xChange")) {
    		//read in the two Qubits that will be exchanged
    		int a = con.getAffectedQubits().get(0);
    		int b = con.getAffectedQubits().get(1);
    		result = exchangeQubits(a, b, state);
    	} else if (operation.startsWith("R_") || operation.startsWith("S_") ||
    			operation.startsWith("Rx(") ||  operation.startsWith("Ry(")  
		 || operation.startsWith("Rz(")) {
    		result = applyMatrix(con.getControllQubits(), con.getAffectedQubits(),state, con.getMatrix());
    	} else if (operation.equals("!")) {
    		// partial measurement
    		result = partialMeasurement(con.getAffectedQubits().get(0), state);	
    	}
    	return result;
    }
    
    /**
     * Used to perfom a complete measurement
     * @param state
     * @return
     */
    public static LinkedHashSet<Ket> fullMeasurement (LinkedHashSet<Ket> state) {
    	// In order to simulate a draw we sum up all possibilities to the point
    	// were they get bigger than a random magnitude 
    	LinkedHashSet<Ket> result = new LinkedHashSet<Ket>();
    	double random = Math.random();
    	Iterator<Ket> it = state.iterator();
    	double sumProb = 0.0;
    	while (it.hasNext()) {
    		Ket workingKet = it.next();
    		try {
				sumProb += Double.parseDouble(MapleConnector.mapleEngine.evaluate("evalf((" + workingKet.getFrontFactor() + ")^2):").toString());
			} catch (NumberFormatException e) {
				e.printStackTrace();
			} catch (MapleException e) {
				e.printStackTrace();
			}
			if (sumProb >= random) {
				Ket temp = workingKet.copy();
				temp.setFrontFactor("1");
				result.add(temp);
				return result;
			}
			if (!it.hasNext()) {
				Ket temp = workingKet.copy();
				temp.setFrontFactor("1");
				result.add(temp);
				return result;
			}
    	}
    	return result;
    }
    
    /**
     * Performs a partial measurement
     * @param effectedQubit The qubit that shall be measured
     * @param state The state that shall be measured
     * @return The measured state
     */
    
    public static LinkedHashSet<Ket> partialMeasurement(int effectedQubit, LinkedHashSet<Ket> state ) {
    	LinkedHashSet<Ket> result = new LinkedHashSet<Ket>();
    	// First we build the probabilities for 0 and 1 in the effected Qubit
    	double probZero = 0;
    	Iterator<Ket> it = state.iterator();
    	while (it.hasNext()) {
    		Ket workingKet = it.next();
    		if (workingKet.getQubitSequenz()[effectedQubit] == 0) {
    			try {
					probZero += Double.parseDouble(MapleConnector.mapleEngine.evaluate("evalf((" + workingKet.getFrontFactor() + ")^2):").toString());
				} catch (NumberFormatException e) {
					e.printStackTrace();
				} catch (MapleException e) {
					e.printStackTrace();
				}
    		}
    	}
    	// Now that we know the probabilities we draw a random number that will decide
    	// whether we choose 0 or 1
    	double rand = Math.random();
    	int decision;
    	if (probZero > rand) {
    		decision = 0;
    	} else {
    		decision = 1;
    	}
    	
    	// Now we have to merge and normalize
    	LinkedHashMap<String, Ket> merger = new LinkedHashMap<String, Ket>();
    	it = state.iterator();
    	while (it.hasNext()) {
    		Ket workingKet = it.next();
    		Ket temp = workingKet.copy();
    		temp.getQubitSequenz()[effectedQubit] = decision;
    		if (!merger.containsKey(makeComparable(temp.getQubitSequenz()))) {
    			merger.put(makeComparable(temp.getQubitSequenz()),temp);
    		} else {
    			// we have to merge
    			// take the sqrt(Ket1^2+Ket2^2)
    			String prevFrontFactor = merger.get(makeComparable(temp.getQubitSequenz())).getFrontFactor();
    			String newFrontFactor="";
    			try {
					newFrontFactor = MapleConnector.mapleEngine.evaluate("evalm(sqrt((" + prevFrontFactor + ")^2 + (" + temp.getFrontFactor() +  ")^2 )):").toString();
				} catch (MapleException e) {
					e.printStackTrace();
				}
				temp.setFrontFactor(newFrontFactor);
				merger.put(makeComparable(temp.getQubitSequenz()), temp);
    			
    		}
    	}
    	result.addAll(merger.values());
    	
    	return result;
    }
    
    /**
     * Converts a jaQuzzi Matrix to a String[][] array
     * @param m The matrix to convert
     * @return The Matrix as a String[][] array
     */
    
    public static String[][] toStringMatrix( Matrix m) {
    	String[][] result=null;
    	return result;
    }
    
}
